C      *****************************************************************
C      * Subroutine CalcResidual(MaxError, MaxComp, LNum)              *
C      * The residual for each component is calculated by taking the   *
C      * difference between the known total concentration and the total*
C      * concentration calculated as a result of mass balance.  The    *
C      * known total concentration is calculated from the known total  *
C      * (KT).  The calculated total concentration is calculated from  *
C      * the current guess at the component concentrations (SpConc).   *
C      * The difference will be  the residual, Resid (one value for    *
C      * each component).  If Resid is negative SpConc is too high.  If*
C      * Resid is postive, SpConc is too low.                          *
C      * This routine also returns an error term which is the ratio of *
C      * the residual to the known concentration.  The largest error   *
C      * term and its corresponding component are returned.            *
C      *                                                               *
C      * Variables:                                                    *
C      * C, C1     -Local - Array indices.  Unitless.                  *
C      * Comp      -Local - Component index.  Unitless.                *
C      * CtoM      -Fcn   - Multiplication factor for conversion from  *
C      *                    concentration to mass.  Units are L.       *
C      * FixAct    -Input - Indicates a species with fixed activity,   *
C      *                    i.e., there is a known concentration of    *
C      *                    this component.  Unitless.                 *
C      *                    (Common block VParam, file VModel.f)       *
C      * KT        -Input - A matrix, of size NComp by NLayer, of the  *
C      *                    known total amount for each component, for *
C      *                    each layer.  The total includes the amount *
C      *                    of the component as a free component and as*
C      *                    part of all dependent species.  Units are  *
C      *                    mol.                                       *
C      *                    (Common block VModel, file VModel.f)       *
C      * LNum      -Input - The current layer number.  Unitless.       *
C      * MasBal    -Input - Indicates a species that is limited by     *
C      *                    mass balance.  Unitless.                   *
C      *                    (Common block VModel, file VModel.f)       *
C      * MaxEl     -Local - Absolute value for residual errors.  Units *
C      *                    are mol.                                   *
C      * MaxError  -Output- Largest ratio of residual to known         *
C      *                    concentration.  Units are mol/mol.         *
C      * MaxComp   -Output- Component index corresponding to MaxError. *
C      *                    Unitless.                                  *
C      * NComp     -Input - The number of components in the system.    *
C      *                    Unitless.                                  *
C      *                    (Common block VModel, file VModel.f)       *
C      * NCompSize -Input - The max number of components, used to size *
C      *                    arrays.  Unitless.                         *
C      *                    (file ArraySizes.Inc)                      *
C      * NPhases   -Input - The number of phases in the system.        *
C      *                    Unitless.                                  *
C      *                    (Common block VSolidPhase, file VModel.f)  *
C      * NSpecies  -Input - The number of species in the system (i.e.  *
C      *                    the number of components plus the number   *
C      *                    of dependent species).  Unitless.          *
C      *                    (Common block VModel, file VModel.f)       *
C      * OutResid  -Local - A vector, of size NComp, of the residual   *
C      *                    for each component. This residual is the   *
C      *                    difference between the known total         *
C      *                    concentration and the total concentration  *
C      *                    calculated from a set of component         *
C      *                    concentrations.  Units are mol.            *
C      * P,P2      -Local - Array indices.  Unitless.                  *
C      * PMass     -Output- A matrix, of size NPhases by NLayers, of   *
C      *                    the amount of each phase dissolved or      *
C      *                    precipitated, per layer.  Units are mol.   *
C      *                    (Common block VSolidPhase, file VModel.f)  *
C      * POrder    -Input - A vector, of size NPhases, of order in     *
C      *                    which to consider phases in mole balance   *
C      *                    calculations.  Unitless.                   *
C      *                    (Common block VSolidPhase, file VModel.f)  *
C      * Q,Q2      -Local - Array indices.  Unitless.                  *
C      * Resid     -Output- A vector, of size NComp, of the residual   *
C      *                    for each component.  This residual is the  *
C      *                    difference between the known total         *
C      *                    concentration and the total concentration  *
C      *                    calculated from a set of component         *
C      *                    concentrations.  Units are mol.            *
C      *                    (Common block VModel, file VModel.f)       *
C      * S         -Local - Array index.  Unitless.                    *
C      * SC        -Input - A matrix, of size NDepSp by NComp, of the  *
C      *                    stoichiometric coefficient of each         *
C      *                    component for each species.  Unitless.     *
C      *                    (Common block VModel, file VModel.f)       *
C      * SComp     -Input - A vector, of size NPhases, of the          *
C      *                    component associated with each phase.      *
C      *                    Unitless.                                  *
C      *                    (Common block VSolidPhase, file VModel.f)  *
C      * SpConc    -Input - A matrix, of size NSpecies by NLayers, of  *
C      *                    species concentrations, one set for each   *
C      *                    layer. Units are mol/L.                    *
C      *                    (Common block VModel, file VModel.f)       *
C      * SPMasBal  -Input - Indicates a solid phase species type       *
C      *                    limited by mass balance.  Unitless.        *
C      *                    (Common block VModel, file VModel.f)       *
C      * SPFixAct  -Input - Indicates a solid phase species type with  *
C      *                    fixed activity, i.e.,  there is a known    *
C      *                    concentration of this species.  Unitless.  *
C      *                    (Common block VParam, file VModel.f)       *
C      * SPSolGas  -Input - Indicates a solid phase species type that  *
C      *                    is solid or gaseous controlled.  Unitless. *
C      *                    (Common block VParam, file VModel.f)       *
C      * SType     -Input - A vector, of size NSpecies, of each        *
C      *                    species's type.  Unitless.                 *
C      *                    (Common block VModel, file VModel.f)       *
C      * Sum       -Local - A sum variable.  Units are consistent with *
C      *                    items accumulated.                         *
C      * T1        -Local - Temporary storage.  Units are consistent   *
C      *                    with item stored.                          *
C      * YASOut    -Input - The output file number for the YASEQL      *
C      *                    model.  Unitless.                          *
C      *                    (Common block VModel, file VModel.f)       *
C      *****************************************************************
       SUBROUTINE CALCRESIDUAL(MAXERROR, MAXCOMP, LNUM, OUTRESID)
							IMPLICIT NONE
							INCLUDE 'VMODEL.INC'
							REAL*8 CT, CTOM, MAXEL, MAXERROR, OUTRESID(NCOMPSIZE)
       REAL*8 SUM, T1
       INTEGER C, C1
							INTEGER LNUM, MAXCOMP, P, P2, Q, Q2, S 

C      *--------------------------------------------------------------*
C      * Calculate PMass - the mass dissolved from or precipitated to *
C      * each phase.                                                  *
C      *--------------------------------------------------------------*
       DO 120 Q = 1, NPHASES
          P = PORDER(Q)
          C1 = SCOMP(P)
          SUM = SPCONC(C1,LNUM) * CTOM(C1,LNUM)
          DO 100 S = NCOMP+1, NSPECIES
             SUM = SUM + SPCONC(S, LNUM) * SC(S, C1) * CTOM(S, LNUM)
  100     CONTINUE

C         *------------------------------------------------*
C         * Contributions from other phases to this phase. *
C         *------------------------------------------------*
          DO 110 Q2 = 1, Q-1
             P2 = PORDER(Q2)
             SUM = SUM + PMASS(P2,LNUM) * SP(P2,C1)
  110     CONTINUE
          PMASS(P,LNUM) = (KT(C1,LNUM) - SUM) / SP(P,C1)
  120  CONTINUE

C      *------------------------*
C      * Begin with components. *
C      *------------------------*
       MAXERROR = 0
       MAXCOMP = 1
       DO 150 C = 1, NCOMP
C         *-----------------------------------------------------------------*
C         * Calculate a mass balance for each component as a sum of species *
C         * concentrations and keep a running sum for error checking.       *
C         *-----------------------------------------------------------------*
          SUM = 0
          MAXEL = 0
          DO 130 S = 1, NSPECIES
             T1 = SPCONC(S,LNUM) * SC(S,C) * CTOM(S,LNUM)
             SUM = SUM + T1
             MAXEL = MAXEL + ABS(T1)
  130     CONTINUE

C         *----------------------------------*
C         * Add in contribution from phases. *
C         *----------------------------------*
          DO 140 P = 1, NPHASES
             C1 = SCOMP(P)
             IF (SP(P,C1).NE.0) THEN
                SUM = SUM + PMASS(P,LNUM) * SP(P,C) / SP(P,C1)
             ELSE
                WRITE(6,*) 'Error in routine CalcResidual'
                WRITE(6,141) 'SP(',P,',',C1,') = 0'
                WRITE(YASOUT,*) 'Error in routine CalcResidual'
                WRITE(YASOUT,141) 'SP(',P,',',C1,') = 0'
  141           FORMAT(' ',A3,I2,A1,I2,A5)
             ENDIF
  140     CONTINUE

C         *-------------------------------------------------------------*
C         * For components limited by mass balance, the residual equals *
C         * the difference between the calculated total and the known   *
C         * total.  For components limited by fixed activity the known  *
C         * becomes the calculated total and therefore there is no      *
C         * residual.  For components solid or gaseous controlled the   *
C         * residual is zero and the known total remains as it is.      *
C         *-------------------------------------------------------------*
          IF ((STYPE(C).EQ.MASBAL).OR.(STYPE(C).EQ.SPMASBAL)) THEN
             RESID(C) = SUM - KT(C,LNUM)
          ELSEIF ((STYPE(C).EQ.FIXACT).OR.(STYPE(C).EQ.SPFIXACT)) THEN
             KT(C,LNUM) = SUM
             RESID(C) = 0
          ELSEIF ((STYPE(C).EQ.SOLGAS).OR.(STYPE(C).EQ.SPSOLGAS)) THEN
             RESID(C) = 0
          ENDIF

C         *----------------------------------------------------*
C         * Check error for the mass balance of this component *
C         * as the residual divided by the sum of the absolute *
C         * values of the species concentrations.  Store       *
C         * largest error among all components to check against*
C         * convergence criterion.                             *
C         *----------------------------------------------------*
C---          IF (MAXEL.NE.0) THEN
          IF (MAXEL.GT.1E-100) THEN
             T1 = ABS(RESID(C)/MAXEL)
             IF (T1.GT.MAXERROR) THEN
                MAXERROR = T1
                MAXCOMP = C
             ENDIF
          ENDIF
  150  CONTINUE

C      *---------------------------------------------------------------*
C      * Set up an array of residuals to pass back to calling routine. *
C      *---------------------------------------------------------------*
       DO 200 C = 1, NCOMP
          OUTRESID(C) = RESID(C)
  200  CONTINUE

       RETURN
	      END
C      *****************************************************************
C      *                    END SUBROUTINE                             *
C      *****************************************************************
